// ===== Romaji -> Hiragana (minimal) =====
const ROMAJI_TABLE = [
  ["kya","きゃ"],["kyu","きゅ"],["kyo","きょ"],
  ["gya","ぎゃ"],["gyu","ぎゅ"],["gyo","ぎょ"],
  ["sha","しゃ"],["shu","しゅ"],["sho","しょ"],
  ["sya","しゃ"],["syu","しゅ"],["syo","しょ"],
  ["cha","ちゃ"],["chu","ちゅ"],["cho","ちょ"],
  ["tya","ちゃ"],["tyu","ちゅ"],["tyo","ちょ"],
  ["nya","にゃ"],["nyu","にゅ"],["nyo","にょ"],
  ["hya","ひゃ"],["hyu","ひゅ"],["hyo","ひょ"],
  ["mya","みゃ"],["myu","みゅ"],["myo","みょ"],
  ["rya","りゃ"],["ryu","りゅ"],["ryo","りょ"],
  ["bya","びゃ"],["byu","びゅ"],["byo","びょ"],
  ["pya","ぴゃ"],["pyu","ぴゅ"],["pyo","ぴょ"],
  ["ja","じゃ"],["ju","じゅ"],["jo","じょ"],

  ["ka","か"],["ki","き"],["ku","く"],["ke","け"],["ko","こ"],
  ["sa","さ"],["shi","し"],["si","し"],["su","す"],["se","せ"],["so","そ"],
  ["ta","た"],["chi","ち"],["ti","ち"],["tsu","つ"],["tu","つ"],["te","て"],["to","と"],
  ["na","な"],["ni","に"],["nu","ぬ"],["ne","ね"],["no","の"],
  ["ha","は"],["hi","ひ"],["fu","ふ"],["hu","ふ"],["he","へ"],["ho","ほ"],
  ["ma","ま"],["mi","み"],["mu","む"],["me","め"],["mo","も"],
  ["ya","や"],["yu","ゆ"],["yo","よ"],
  ["ra","ら"],["ri","り"],["ru","る"],["re","れ"],["ro","ろ"],
  ["wa","わ"],["wo","を"],
  ["ga","が"],["gi","ぎ"],["gu","ぐ"],["ge","げ"],["go","ご"],
  ["za","ざ"],["ji","じ"],["zi","じ"],["zu","ず"],["ze","ぜ"],["zo","ぞ"],
  ["da","だ"],["de","で"],["do","ど"],
  ["ba","ば"],["bi","び"],["bu","ぶ"],["be","べ"],["bo","ぼ"],
  ["pa","ぱ"],["pi","ぴ"],["pu","ぷ"],["pe","ぺ"],["po","ぽ"],

  ["a","あ"],["i","い"],["u","う"],["e","え"],["o","お"]
];

function isLikelyRomajiWord(s) {
  if (!s) return false;
  if (s.length > 80) return false;
  // 記号が多いものやURL/メールっぽいものは除外（必要なら緩めてOK）
  if (/[0-9]/.test(s)) return false;
  if (/[\/:@.#?&=]/.test(s)) return false;
  return /^[a-zA-Z'\-]+$/.test(s);
}

function romajiToHiragana(input) {
  let s = input.toLowerCase();

  // 小さい「っ」: 子音の連続
  s = s.replace(/([bcdfghjklmpqrstvwxyz])\1/g, "っ$1");
  // n -> ん（簡易）
  s = s.replace(/n(?=[^aeiouy]|$)/g, "ん");

  let out = s;
  ROMAJI_TABLE
    .slice()
    .sort((a,b) => b[0].length - a[0].length)
    .forEach(([ro, hi]) => {
      out = out.split(ro).join(hi);
    });

  return out;
}

function isEditableTarget(el) {
  if (!el) return false;
  const tag = el.tagName?.toLowerCase();
  if (tag === "textarea") return true;
  if (tag === "input") {
    const type = (el.getAttribute("type") || "text").toLowerCase();
    // 検索バー対応：search も許可。URL/emailも一応OK（嫌なら外してOK）
    const ok = ["text","search","url","email","tel"].includes(type);
    return ok;
  }
  if (el.isContentEditable) return true;
  return false;
}

// ====== Space/Enterで直前単語を変換（既存機能） ======
function replaceLastWordInInput(el) {
  const value = el.value;
  const pos = el.selectionStart ?? value.length;
  const before = value.slice(0, pos);
  const after = value.slice(pos);

  const m = before.match(/([A-Za-z'\-]+)$/);
  if (!m) return;
  const word = m[1];
  if (!isLikelyRomajiWord(word)) return;

  const hira = romajiToHiragana(word);
  const start = before.length - word.length;
  const newBefore = before.slice(0, start) + hira;

  el.value = newBefore + after;
  const newPos = newBefore.length;
  el.setSelectionRange(newPos, newPos);
}

function replaceLastWordInContentEditable() {
  const sel = window.getSelection();
  if (!sel || sel.rangeCount === 0) return;
  const range = sel.getRangeAt(0);
  if (!range.collapsed) return;

  const node = range.startContainer;
  if (!node || node.nodeType !== Node.TEXT_NODE) return;

  const text = node.nodeValue ?? "";
  const offset = range.startOffset;

  const before = text.slice(0, offset);
  const after = text.slice(offset);

  const m = before.match(/([A-Za-z'\-]+)$/);
  if (!m) return;
  const word = m[1];
  if (!isLikelyRomajiWord(word)) return;

  const hira = romajiToHiragana(word);
  const start = before.length - word.length;

  node.nodeValue = before.slice(0, start) + hira + after;

  const newOffset = (before.slice(0, start) + hira).length;
  const newRange = document.createRange();
  newRange.setStart(node, newOffset);
  newRange.collapse(true);
  sel.removeAllRanges();
  sel.addRange(newRange);
}

document.addEventListener("keydown", (e) => {
  if (!(e.key === " " || e.key === "Enter")) return;
  if (e.isComposing) return;

  const el = document.activeElement;
  if (!isEditableTarget(el)) return;

  try {
    if (el.isContentEditable) replaceLastWordInContentEditable();
    else replaceLastWordInInput(el);
  } catch (_) {}
}, true);

// ====== ドラッグ選択→ボタンで変換 ======
let floatingBtn = null;
let lastSelectionInfo = null;

function ensureButton() {
  if (floatingBtn) return;
  floatingBtn = document.createElement("button");
  floatingBtn.textContent = "変換";
  floatingBtn.type = "button";
  floatingBtn.style.position = "fixed";
  floatingBtn.style.zIndex = "2147483647";
  floatingBtn.style.padding = "6px 10px";
  floatingBtn.style.fontSize = "12px";
  floatingBtn.style.borderRadius = "10px";
  floatingBtn.style.border = "1px solid #ccc";
  floatingBtn.style.background = "#fff";
  floatingBtn.style.boxShadow = "0 2px 10px rgba(0,0,0,0.15)";
  floatingBtn.style.display = "none";

  floatingBtn.addEventListener("mousedown", (e) => e.preventDefault()); // 選択解除を防ぐ
  floatingBtn.addEventListener("click", () => {
    if (!lastSelectionInfo) return;
    applySelectionConversion(lastSelectionInfo);
    hideButton();
  });

  document.documentElement.appendChild(floatingBtn);
}

function hideButton() {
  if (!floatingBtn) return;
  floatingBtn.style.display = "none";
  lastSelectionInfo = null;
}

function showButtonAt(x, y) {
  ensureButton();
  floatingBtn.style.left = `${Math.min(x + 8, window.innerWidth - 80)}px`;
  floatingBtn.style.top = `${Math.min(y + 8, window.innerHeight - 40)}px`;
  floatingBtn.style.display = "block";
}

function getInputSelectionInfo(el) {
  const start = el.selectionStart;
  const end = el.selectionEnd;
  if (start == null || end == null || start === end) return null;

  const selected = el.value.slice(start, end);
  const trimmed = selected.trim();
  if (!isLikelyRomajiWord(trimmed)) return null;

  return { kind: "input", el, start, end, selected, trimmed };
}

function getContentEditableSelectionInfo() {
  const sel = window.getSelection();
  if (!sel || sel.rangeCount === 0) return null;

  const range = sel.getRangeAt(0);
  if (range.collapsed) return null;

  const selected = sel.toString();
  const trimmed = selected.trim();
  if (!isLikelyRomajiWord(trimmed)) return null;

  // 範囲が複雑すぎると壊れるので、最初はテキストノード内だけ対応
  if (range.startContainer !== range.endContainer) return null;
  if (range.startContainer.nodeType !== Node.TEXT_NODE) return null;

  return { kind: "contenteditable", range, selected, trimmed };
}

function applySelectionConversion(info) {
  const hira = romajiToHiragana(info.trimmed);

  if (info.kind === "input") {
    const before = info.el.value.slice(0, info.start);
    const after = info.el.value.slice(info.end);
    // 前後の空白は維持しつつ、中心だけ変換
    const replaced = info.selected.replace(info.trimmed, hira);
    info.el.value = before + replaced + after;

    const newPos = before.length + replaced.length;
    info.el.setSelectionRange(newPos, newPos);
    return;
  }

  if (info.kind === "contenteditable") {
    const r = info.range;
    const textNode = r.startContainer;
    const full = textNode.nodeValue ?? "";
    const before = full.slice(0, r.startOffset);
    const mid = full.slice(r.startOffset, r.endOffset);
    const after = full.slice(r.endOffset);

    const replaced = mid.replace(info.trimmed, hira);
    textNode.nodeValue = before + replaced + after;

    const sel = window.getSelection();
    if (sel) {
      const newOffset = before.length + replaced.length;
      const newRange = document.createRange();
      newRange.setStart(textNode, newOffset);
      newRange.collapse(true);
      sel.removeAllRanges();
      sel.addRange(newRange);
    }
  }
}

document.addEventListener("mouseup", (e) => {
  const el = document.activeElement;
  if (!isEditableTarget(el)) {
    hideButton();
    return;
  }

  // input/textarea
  if (el.tagName && (el.tagName.toLowerCase() === "input" || el.tagName.toLowerCase() === "textarea")) {
    const info = getInputSelectionInfo(el);
    if (!info) {
      hideButton();
      return;
    }
    lastSelectionInfo = info;
    showButtonAt(e.clientX, e.clientY);
    return;
  }

  // contenteditable
  if (el.isContentEditable) {
    const info = getContentEditableSelectionInfo();
    if (!info) {
      hideButton();
      return;
    }
    lastSelectionInfo = info;
    showButtonAt(e.clientX, e.clientY);
    return;
  }
}, true);

// クリックで消す
document.addEventListener("mousedown", (e) => {
  if (floatingBtn && e.target === floatingBtn) return;
  hideButton();
}, true);

// Escで消す
document.addEventListener("keydown", (e) => {
  if (e.key === "Escape") hideButton();
}, true);
